﻿#Requires -Version 5.1
<#
.SYNOPSIS
    Veeam Backup & Replication 12.3 - VM Protection HTML Report Generator

.DESCRIPTION
    Connects to a local Veeam B&R server, enumerates all backup jobs, and generates
    a comprehensive HTML report showing:
      - VMs protected by each backup job
      - Destination repository
      - Schedule configuration
      - Retention policy
      - Current protection status (last session result)
      - Non-standard / notable job configuration settings

.NOTES
    Requirements:
      - Veeam Backup & Replication 12.3 installed locally
      - Run as a user with Veeam Administrator or Read-Only Administrator rights
      - Veeam PowerShell snap-in (VeeamPSSnapIn) must be available

.EXAMPLE
    .\VeeamBackupReport.ps1
    .\VeeamBackupReport.ps1 -OutputPath "C:\Reports\VeeamReport.html" -IncludeDisabledJobs
#>

[CmdletBinding()]
param(
    [string]$OutputPath = "",
    [switch]$IncludeDisabledJobs,
    [switch]$IncludeAgentJobs
)

#region ── Helper Functions ────────────────────────────────────────────────────

function Load-VeeamSnapin {
    # Veeam 12.x ships as a PowerShell module (Veeam.Backup.PowerShell).
    # The legacy Add-PSSnapin (VeeamPSSnapIn) is no longer registered in v12.

    $moduleName = "Veeam.Backup.PowerShell"

    if (Get-Module -Name $moduleName) {
        Write-Host "[OK] Veeam PowerShell module already loaded." -ForegroundColor Green
        return
    }

    # Try standard Import-Module first (works if module is on PSModulePath)
    if (Import-Module $moduleName -ErrorAction SilentlyContinue -PassThru) {
        Write-Host "[OK] Veeam PowerShell module loaded." -ForegroundColor Green
        return
    }

    # Fallback: locate the module from the Veeam install directory in the registry
    $regPaths = @(
        "HKLM:\SOFTWARE\Veeam\Veeam Backup and Replication",
        "HKLM:\SOFTWARE\WOW6432Node\Veeam\Veeam Backup and Replication"
    )
    $installDir = $null
    foreach ($rp in $regPaths) {
        if (Test-Path $rp) {
            $installDir = (Get-ItemProperty $rp -ErrorAction SilentlyContinue).CorePath
            if (-not $installDir) {
                $installDir = (Get-ItemProperty $rp -ErrorAction SilentlyContinue).InstallationPath
            }
            if ($installDir) { break }
        }
    }

    if ($installDir) {
        $candidates = @(
            (Join-Path $installDir "Packages\VeeamPSSnapIn\Veeam.Backup.PowerShell.dll"),
            (Join-Path $installDir "Veeam.Backup.PowerShell.dll"),
            (Join-Path $installDir "Packages\VeeamPSSnapIn\Veeam.Backup.PowerShell.psd1"),
            (Join-Path $installDir "Veeam.Backup.PowerShell.psd1")
        )
        foreach ($c in $candidates) {
            if (Test-Path $c) {
                try {
                    Import-Module $c -ErrorAction Stop
                    Write-Host "[OK] Veeam PowerShell module loaded from: $c" -ForegroundColor Green
                    return
                } catch {
                    Write-Warning "Found $c but failed to import: $_"
                }
            }
        }
    }

    # Last resort: search common install locations
    $searchRoots = @(
        "$env:ProgramFiles\Veeam\Backup and Replication\Console",
        "$env:ProgramFiles\Veeam\Backup and Replication\Backup",
        "$env:ProgramFiles\Veeam\Backup and Replication"
    )
    foreach ($root in $searchRoots) {
        if (Test-Path $root) {
            $found = Get-ChildItem -Path $root -Filter "Veeam.Backup.PowerShell.dll" -Recurse -ErrorAction SilentlyContinue | Select-Object -First 1
            if ($found) {
                try {
                    Import-Module $found.FullName -ErrorAction Stop
                    Write-Host "[OK] Veeam PowerShell module loaded from: $($found.FullName)" -ForegroundColor Green
                    return
                } catch {
                    Write-Warning "Found $($found.FullName) but failed to import: $_"
                }
            }
        }
    }

    Write-Error "Could not load the Veeam PowerShell module ($moduleName). Ensure Veeam B&R 12.3 is installed and run PowerShell as Administrator."
    exit 1
}

function Connect-VeeamServer {
    try {
        $connection = Get-VBRServerSession
        if ($null -eq $connection) {
            Connect-VBRServer -Server localhost -ErrorAction Stop
        }
        Write-Host "[OK] Connected to local Veeam server." -ForegroundColor Green
    }
    catch {
        Write-Error "Could not connect to local Veeam server: $_"
        exit 1
    }
}

function Get-ScheduleDescription {
    param([object]$Job)

    if (-not $Job.IsScheduleEnabled) { return "<span class='badge badge-disabled'>Not Scheduled</span>" }

    $opts  = $Job.ScheduleOptions
    $parts = @()

    # --- After Job ---
    # The parent job ID is not exposed on the child job's ScheduleOptions.
    # Instead, scan ALL jobs to find which one lists THIS job as its linked/chained target.
    # Veeam sets LinkedJob on the PARENT side, pointing to the child that runs after it.
    if ($opts.OptionsScheduleAfterJob.IsEnabled) {
        $linkedName = $null

        try {
            $allJobs = Get-VBRJob -ErrorAction SilentlyContinue
            foreach ($candidate in $allJobs) {
                if ($candidate.Id.ToString() -eq $Job.Id.ToString()) { continue }
                try {
                    # Try .NextJobId property on the parent job
                    $nextId = $candidate.NextJobId
                    if ($nextId -and $nextId.ToString() -eq $Job.Id.ToString()) {
                        $linkedName = $candidate.Name; break
                    }
                } catch {}
                try {
                    # Try .Info.LinkedJobId
                    $nextId = $candidate.Info.LinkedJobId
                    if ($nextId -and $nextId.ToString() -eq $Job.Id.ToString()) {
                        $linkedName = $candidate.Name; break
                    }
                } catch {}
                try {
                    # Try .Options.JobOptions.LinkedJobId
                    $nextId = $candidate.Options.JobOptions.LinkedJobId
                    if ($nextId -and $nextId.ToString() -eq $Job.Id.ToString()) {
                        $linkedName = $candidate.Name; break
                    }
                } catch {}
                try {
                    # Try enumerating all properties dynamically for any that hold this job's GUID
                    $thisId = $Job.Id.ToString()
                    $candidate | Get-Member -MemberType Property -ErrorAction SilentlyContinue | ForEach-Object {
                        if ($linkedName) { return }
                        try {
                            $val = $candidate.($_.Name)
                            if ($val -and $val.ToString() -eq $thisId) {
                                $linkedName = $candidate.Name
                            }
                        } catch {}
                    }
                } catch {}
                if ($linkedName) { break }
            }
        } catch {}

        if ($linkedName) {
            $parts += "After job: <em>$([System.Net.WebUtility]::HtmlEncode($linkedName))</em>"
        } else {
            $parts += "After previous job"
        }
    }

    # --- Daily ---
    # Confirmed via diagnostic: .Enabled (not .IsEnabled), .Kind, .TimeLocal
    if ($opts.OptionsDaily.Enabled) {
        try {
            if ($opts.OptionsDaily.Kind.ToString() -eq "Everyday") {
                $days = "Every day"
            } else {
                $days = ($opts.OptionsDaily.DaysSrv | ForEach-Object { $_.ToString().Substring(0,3) }) -join ', '
                if (-not $days) { $days = "Selected days" }
            }
            $time = $opts.OptionsDaily.TimeLocal.ToString("HH:mm")
            $parts += "Daily [$days] at $time"
        } catch { $parts += "Daily" }
    }

    # --- Periodically ---
    # Confirmed: .Enabled, .FullPeriod in seconds, .Kind = Hours or Minutes
    if ($opts.OptionsPeriodically.Enabled) {
        try {
            $secs = [int]$opts.OptionsPeriodically.FullPeriod
            if ($opts.OptionsPeriodically.Kind.ToString() -eq "Hours" -or $secs -ge 3600) {
                $hrs = [Math]::Round($secs / 3600, 1)
                $parts += "Every $hrs hour(s)"
            } else {
                $mins = [Math]::Round($secs / 60, 0)
                $parts += "Every $mins minute(s)"
            }
        } catch { $parts += "Periodically" }
    }

    # --- Monthly ---
    # Confirmed: .Enabled, .DayNumberInMonth, .TimeLocal
    if ($opts.OptionsMonthly.Enabled) {
        try {
            $day  = $opts.OptionsMonthly.DayNumberInMonth
            $time = $opts.OptionsMonthly.TimeLocal.ToString("HH:mm")
            $parts += "Monthly ($day) at $time"
        } catch { $parts += "Monthly" }
    }

    # --- Continuous ---
    # Confirmed: IsContinuous is a direct bool on ScheduleOptions itself
    if ($opts.IsContinuous -or $opts.OptionsContinuous.Enabled) {
        $parts += "<strong>Continuous</strong>"
    }

    if ($parts.Count -eq 0) { return "Scheduled (manual trigger)" }
    return $parts -join "<br>"
}

function Get-RetentionDescription {
    param([object]$Job)
    try {
        $opts = $Job.BackupStorageOptions
        $rp   = $opts.RetainCycles
        $unit = "restore points"

        # GFS
        $gfs = $opts.EnableDeletedVmDataRetention
        $gfsParts = @()
        if ($opts.GFSWeeklyKeepBackups -gt 0)  { $gfsParts += "Weekly: $($opts.GFSWeeklyKeepBackups)w" }
        if ($opts.GFSMonthlyKeepBackups -gt 0) { $gfsParts += "Monthly: $($opts.GFSMonthlyKeepBackups)m" }
        if ($opts.GFSYearlyKeepBackups -gt 0)  { $gfsParts += "Yearly: $($opts.GFSYearlyKeepBackups)y" }

        $result = "$rp $unit"
        if ($gfsParts.Count -gt 0) { $result += "<br><em>GFS: $($gfsParts -join ', ')</em>" }
        return $result
    }
    catch { return "N/A" }
}

function Get-LastSessionStatus {
    param([object]$Job)
    try {
        $session = Get-VBRBackupSession -Job $Job -ErrorAction SilentlyContinue |
                   Sort-Object EndTimeUTC -Descending |
                   Select-Object -First 1

        if ($null -eq $session) { return @{ Badge = "<span class='badge badge-none'>No Sessions</span>"; Time = "Never" } }

        $result = $session.Result
        $end    = if ($session.EndTimeUTC -ne [datetime]::MinValue) { $session.EndTimeUTC.ToLocalTime().ToString("yyyy-MM-dd HH:mm") } else { "Running..." }

        $badge = switch ($result) {
            "Success" { "<span class='badge badge-success'>Success</span>" }
            "Warning" { "<span class='badge badge-warning'>Warning</span>" }
            "Failed"  { "<span class='badge badge-failed'>Failed</span>" }
            default   { "<span class='badge badge-none'>$result</span>" }
        }
        return @{ Badge = $badge; Time = $end }
    }
    catch { return @{ Badge = "<span class='badge badge-none'>Unknown</span>"; Time = "N/A" } }
}

function Get-NonStandardSettings {
    param([object]$Job)

    $flags = @()
    try {
        $storage = $Job.BackupStorageOptions
        $vss     = $Job.VSSOptions
        $net     = $Job.NetworkOptions
        $gen     = $Job.Options

        # Compression / dedup
        if ($storage.CompressionLevel -notin @(5)) { # 5 = Optimal (default)
            $levelMap = @{0="None"; 4="Dedupe-friendly"; 5="Optimal"; 6="High"; 9="Extreme"}
            $compLabel = if ($levelMap.ContainsKey($storage.CompressionLevel)) { $levelMap[$storage.CompressionLevel] } else { $storage.CompressionLevel }
            $flags += "Compression: <strong>$compLabel</strong> (default: Optimal)"
        }

        # Storage optimisation
        if ($storage.StorageOptimizationType -ne "LocalTarget") {
            $flags += "Storage Optimisation: <strong>$($storage.StorageOptimizationType)</strong>"
        }

        # Encryption
        if ($storage.StorageEncryptionEnabled) {
            $flags += "<strong>Encryption enabled</strong>"
        }

        # Application-aware processing
        if (-not $vss.Enabled) {
            $flags += "Application-aware processing: <strong>Disabled</strong>"
        }

        # Guest interaction proxy
        if ($vss.GuestProxy -ne $null -and $vss.GuestProxy -ne "") {
            $flags += "Guest proxy: <strong>$($vss.GuestProxy)</strong>"
        }

        # Backup mode (Incremental / Reverse Incremental / Full)
        $mode = $Job.BackupType
        if ($mode -ne "Increment") {
            $flags += "Backup mode: <strong>$mode</strong>"
        }

        # Active full schedule
        $af = $Job.BackupStorageOptions.EnableFullBackup
        if ($af) {
            $flags += "Active Full backup <strong>enabled</strong>"
        }

        # Synthetic full
        $sf = $Job.BackupStorageOptions.SyntheticsEnabled
        if ($sf) {
            $flags += "Synthetic Full <strong>enabled</strong>"
        }

        # Changed block tracking
        try {
            if (-not $Job.ViSourceOptions.EnableChangeTracking) {
                $flags += "CBT (Changed Block Tracking): <strong>Disabled</strong>"
            }
        } catch {}

        # Exclude deleted VMs
        if (-not $storage.EnableDeletedVmDataRetention) {
            $flags += "Deleted VM data retention: <strong>Disabled</strong>"
        }

        # Network throttling
        try {
            if ($net.ThrottlingEnabled) {
                $flags += "Network throttling: <strong>Enabled ($($net.ThrottlingValue) $($net.ThrottlingUnit))</strong>"
            }
        } catch {}

        # Max concurrent tasks (non-default)
        try {
            if ($Job.Options.JobOptions.MaxParallelRepos -gt 0) {
                $flags += "Max parallel repos: <strong>$($Job.Options.JobOptions.MaxParallelRepos)</strong>"
            }
        } catch {}

        # Sealing (immutability)
        try {
            if ($storage.ImmuteEnabled) {
                $flags += "Immutability: <strong>Enabled ($($storage.ImmutePeriod) days)</strong>"
            }
        } catch {}

        # Secondary target (backup copy)
        try {
            $linkedJobs = Get-VBRJobLinks -Job $Job -ErrorAction SilentlyContinue
            if ($linkedJobs) { $flags += "Linked to backup copy job(s)" }
        } catch {}

    }
    catch {
        $flags += "<em>Could not fully enumerate settings: $_</em>"
    }

    if ($flags.Count -eq 0) { return "<span class='muted'>None detected</span>" }
    return "<ul class='flag-list'><li>" + ($flags -join "</li><li>") + "</li></ul>"
}

function Get-VMList {
    param([object]$Job)
    try {
        # Use the job object's own method - correctly scoped to THIS job, no cmdlet ambiguity
        $jobObjects = $Job.GetObjectsInJob()
        if ($null -eq $jobObjects -or @($jobObjects).Count -eq 0) {
            return "<em class='muted'>No objects found</em>"
        }

        # Build a lookup of per-VM task results from the last session
        $taskResultMap = @{}
        try {
            $lastSession = Get-VBRBackupSession -Job $Job -ErrorAction SilentlyContinue |
                           Sort-Object EndTimeUTC -Descending |
                           Select-Object -First 1
            if ($lastSession) {
                $tasks = Get-VBRTaskSession -Session $lastSession -ErrorAction SilentlyContinue
                foreach ($t in $tasks) {
                    $taskResultMap[$t.Name] = $t.Status
                }
            }
        } catch {}

        # Separate included vs excluded objects
        $included = @($jobObjects | Where-Object { -not $_.IsExcluded })
        $excluded  = @($jobObjects | Where-Object { $_.IsExcluded })

        $rows = @()

        foreach ($obj in ($included | Sort-Object Name)) {
            $vmName = [System.Net.WebUtility]::HtmlEncode($obj.Name)
            $rows += "<tr><td class='vm-name'>" + $vmName + "</td></tr>"
        }

        # Excluded objects (greyed out)
        foreach ($obj in ($excluded | Sort-Object Name)) {
            $vmName = [System.Net.WebUtility]::HtmlEncode($obj.Name)
            $rows += "<tr class='excluded-row'><td class='vm-name'>" + $vmName + " <span class='badge badge-disabled'>Excluded</span></td></tr>"
        }

        if ($rows.Count -eq 0) { return "<em class='muted'>No objects found</em>" }

        $tableHtml  = "<table class='vm-table'>"
        $tableHtml += "<thead><tr><th>VM / Object Name</th></tr></thead>"
        $tableHtml += "<tbody>" + ($rows -join "") + "</tbody>"
        $tableHtml += "</table>"
        return $tableHtml
    }
    catch {
        return "<em class='muted'>Could not retrieve objects: $_</em>"
    }
}

function Get-RepositoryName {
    param([object]$Job)
    try {
        $repo = [Veeam.Backup.Core.CBackupRepositoryAccess]::Get($Job.Info.TargetRepositoryId)
        if ($repo) { return $repo.Name }
    } catch {}
    try {
        return $Job.GetTargetRepository().Name
    } catch {}
    return "N/A"
}

#endregion

#region ── HTML Template ───────────────────────────────────────────────────────

function Build-HTML {
    param(
        [string]$Body,
        [string]$ServerName,
        [string]$ReportDate,
        [int]$TotalJobs,
        [int]$TotalVMs
    )

    return @"
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Veeam Backup Protection Report</title>
<style>
  :root {
    --primary:   #007BC1;
    --primary-dk:#005a8e;
    --bg:        #f4f6f9;
    --card:      #ffffff;
    --border:    #dde3ec;
    --text:      #1e2a38;
    --muted:     #6b7a8d;
    --success:   #2e7d32;
    --warning:   #e65100;
    --failed:    #b71c1c;
    --disabled:  #616161;
  }
  *, *::before, *::after { box-sizing: border-box; margin: 0; padding: 0; }
  body { font-family: 'Segoe UI', Arial, sans-serif; background: var(--bg); color: var(--text); font-size: 13px; }

  /* ── Header ── */
  .page-header {
    background: linear-gradient(135deg, var(--primary-dk) 0%, var(--primary) 100%);
    color: #fff; padding: 28px 40px;
  }
  .page-header h1 { font-size: 22px; font-weight: 600; letter-spacing: .3px; }
  .page-header .meta { margin-top: 6px; opacity: .85; font-size: 12px; }

  /* ── Summary bar ── */
  .summary-bar {
    display: flex; gap: 16px; flex-wrap: wrap;
    background: #fff; border-bottom: 1px solid var(--border);
    padding: 14px 40px;
  }
  .stat-card {
    background: var(--bg); border-radius: 6px; padding: 10px 20px;
    min-width: 140px; text-align: center;
  }
  .stat-card .num { font-size: 26px; font-weight: 700; color: var(--primary); }
  .stat-card .lbl { font-size: 11px; color: var(--muted); margin-top: 2px; }

  /* ── Content ── */
  .content { padding: 28px 40px; display: flex; flex-direction: column; gap: 24px; }

  /* ── Job card ── */
  .job-card {
    background: var(--card); border: 1px solid var(--border);
    border-radius: 8px; overflow: hidden;
    box-shadow: 0 1px 4px rgba(0,0,0,.06);
  }
  .job-header {
    display: flex; align-items: center; gap: 12px;
    background: #f0f4f8; border-bottom: 1px solid var(--border);
    padding: 12px 20px;
  }
  .job-icon { font-size: 20px; }
  .job-name { font-size: 15px; font-weight: 600; flex: 1; }
  .job-type { font-size: 11px; color: var(--muted); }

  /* ── Info grid inside card ── */
  .info-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 0;
    border-bottom: 1px solid var(--border);
  }
  .info-cell {
    padding: 14px 20px;
    border-right: 1px solid var(--border);
    border-bottom: 1px solid var(--border);
  }
  .info-cell:last-child { border-right: none; }
  .info-label {
    font-size: 10px; font-weight: 600; text-transform: uppercase;
    letter-spacing: .6px; color: var(--muted); margin-bottom: 5px;
  }
  .info-value { line-height: 1.5; }

  /* ── VM list section ── */
  .vm-section { padding: 14px 20px; }
  .vm-section .section-title {
    font-size: 10px; font-weight: 600; text-transform: uppercase;
    letter-spacing: .6px; color: var(--muted); margin-bottom: 8px;
  }
  .vm-list, .flag-list { padding-left: 18px; line-height: 1.8; }
  .vm-list li, .flag-list li { padding: 1px 0; }
  .obj-type { color: var(--muted); font-size: 11px; }

  /* VM table */
  .vm-table { width: 100%; border-collapse: collapse; font-size: 12px; }
  .vm-table thead tr { background: #eef2f7; }
  .vm-table th {
    text-align: left; padding: 7px 12px;
    font-size: 10px; font-weight: 600; text-transform: uppercase;
    letter-spacing: .5px; color: var(--muted);
    border-bottom: 2px solid var(--border);
  }
  .vm-table td { padding: 6px 12px; border-bottom: 1px solid var(--border); vertical-align: middle; }
  .vm-table tbody tr:last-child td { border-bottom: none; }
  .vm-table tbody tr:hover { background: #f7f9fc; }
  .vm-name { font-weight: 500; }
  .vm-type { color: var(--muted); font-size: 11px; white-space: nowrap; }
  .vm-path { color: var(--muted); font-size: 11px; font-style: italic; }
  .excluded-row { opacity: .55; }
  .excluded-row td { font-style: italic; }

  /* ── Badges ── */
  .badge {
    display: inline-block; padding: 2px 8px; border-radius: 12px;
    font-size: 11px; font-weight: 600;
  }
  .badge-success  { background: #e8f5e9; color: var(--success); }
  .badge-warning  { background: #fff3e0; color: var(--warning); }
  .badge-failed   { background: #ffebee; color: var(--failed);  }
  .badge-disabled { background: #eeeeee; color: var(--disabled); }
  .badge-none     { background: #eceff1; color: var(--muted); }

  .muted { color: var(--muted); }

  /* ── Non-standard flags ── */
  .flags-section { padding: 10px 20px 14px; background: #fffde7; border-top: 1px solid #ffe082; }
  .flags-section .section-title { color: #795548; }

  /* ── Footer ── */
  footer {
    text-align: center; padding: 20px;
    color: var(--muted); font-size: 11px;
    border-top: 1px solid var(--border); background: #fff;
  }

  /* ── Disabled job overlay ── */
  .job-card.disabled { opacity: .7; }
  .job-card.disabled .job-header { background: #eeeeee; }

  @media print {
    body { background: #fff; font-size: 11px; }
    .content { padding: 10px; }
    .job-card { page-break-inside: avoid; }
  }
</style>
</head>
<body>

<header class="page-header">
  <h1>&#x1F6E1; Veeam Backup &amp; Replication &mdash; VM Protection Report</h1>
  <div class="meta">
    Server: <strong>$ServerName</strong> &nbsp;|&nbsp;
    Generated: <strong>$ReportDate</strong> &nbsp;|&nbsp;
    Veeam B&amp;R 12.3
  </div>
</header>

<div class="summary-bar">
  <div class="stat-card">
    <div class="num">$TotalJobs</div>
    <div class="lbl">Backup Jobs</div>
  </div>
  <div class="stat-card">
    <div class="num">$TotalVMs</div>
    <div class="lbl">Protected Objects</div>
  </div>
</div>

<div class="content">
$Body
</div>

<footer>
  Generated by VeeamBackupReport.ps1 &mdash; Veeam B&amp;R 12.3 &mdash; $ReportDate
</footer>
</body>
</html>
"@
}

#endregion

#region ── Main ───────────────────────────────────────────────────────────────

Load-VeeamSnapin
Connect-VeeamServer

$reportDate  = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
$serverName  = $env:COMPUTERNAME
$bodyParts   = @()
$totalVMs    = 0

Write-Host "`nCollecting backup jobs..." -ForegroundColor Cyan

# Retrieve VMware / Hyper-V backup jobs
$jobs = Get-VBRJob -ErrorAction SilentlyContinue | Where-Object {
    $_.JobType -in @("Backup", "BackupSync") -and
    ($IncludeDisabledJobs -or $_.IsScheduleEnabled -or $_.FindLastSession() -ne $null)
}

if ($IncludeAgentJobs) {
    $agentJobs = Get-VBRComputerBackupJob -ErrorAction SilentlyContinue
    # Agent jobs have a different object model; basic info only
}

if ($null -eq $jobs -or $jobs.Count -eq 0) {
    Write-Warning "No backup jobs found on this server."
}

$jobCount = 0

foreach ($job in $jobs) {
    $jobCount++
    Write-Host "  Processing job [$jobCount]: $($job.Name)" -ForegroundColor White

    $isDisabled   = -not $job.IsScheduleEnabled
    $disabledClass= if ($isDisabled) { " disabled" } else { "" }
    $disabledBadge= if ($isDisabled) { "<span class='badge badge-disabled'>Disabled</span> " } else { "" }

    # Collect data
    $lastSession  = Get-LastSessionStatus -Job $job
    $vmListHtml   = Get-VMList -Job $job
    $schedHtml    = Get-ScheduleDescription -Job $job
    $retHtml      = Get-RetentionDescription -Job $job
    $repoName     = Get-RepositoryName -Job $job
    $nonStdHtml   = Get-NonStandardSettings -Job $job

    # Count objects using the job's own method - correctly scoped per job
    try {
        $objCount  = @($job.GetObjectsInJob() | Where-Object { -not $_.IsExcluded }).Count
        $totalVMs += $objCount
        $vmCountStr= "$objCount object(s)"
    } catch { $vmCountStr = "N/A" }

    $jobTypeStr = switch ($job.JobType) {
        "Backup"     { "VM Backup" }
        "BackupSync" { "Backup Copy" }
        default      { $job.JobType }
    }

    # Build card HTML
    $card = @"
<div class="job-card$disabledClass">
  <div class="job-header">
    <span class="job-icon">&#x1F4BE;</span>
    <span class="job-name">$disabledBadge$([System.Net.WebUtility]::HtmlEncode($job.Name))</span>
    <span class="job-type">$jobTypeStr &nbsp;|&nbsp; $vmCountStr</span>
  </div>

  <div class="info-grid">
    <div class="info-cell">
      <div class="info-label">Last Session</div>
      <div class="info-value">$($lastSession.Badge)<br><span class='muted'>$($lastSession.Time)</span></div>
    </div>
    <div class="info-cell">
      <div class="info-label">Destination Repository</div>
      <div class="info-value">$([System.Net.WebUtility]::HtmlEncode($repoName))</div>
    </div>
    <div class="info-cell">
      <div class="info-label">Schedule</div>
      <div class="info-value">$schedHtml</div>
    </div>
    <div class="info-cell">
      <div class="info-label">Retention Policy</div>
      <div class="info-value">$retHtml</div>
    </div>
    <div class="info-cell">
      <div class="info-label">Description</div>
      <div class="info-value">$(if ($job.Description) { [System.Net.WebUtility]::HtmlEncode($job.Description) } else { '<span class="muted">—</span>' })</div>
    </div>
  </div>

  <div class="vm-section">
    <div class="section-title">&#x1F5A5; Protected Objects</div>
    $vmListHtml
  </div>

  <div class="flags-section">
    <div class="section-title">&#x26A0; Non-Standard / Notable Settings</div>
    $nonStdHtml
  </div>
</div>
"@
    $bodyParts += $card
}

# Handle agent jobs (summary only — API differs significantly)
if ($IncludeAgentJobs -and $agentJobs) {
    foreach ($aj in $agentJobs) {
        $jobCount++
        $card = @"
<div class="job-card">
  <div class="job-header">
    <span class="job-icon">&#x1F4BB;</span>
    <span class="job-name">$([System.Net.WebUtility]::HtmlEncode($aj.Name))</span>
    <span class="job-type">Agent Backup Job</span>
  </div>
  <div class="vm-section">
    <div class="section-title">Note</div>
    <p>Agent-based jobs use a different API. Enable -IncludeAgentJobs for basic listing. Full detail collection requires additional scripting against the agent job object model.</p>
  </div>
</div>
"@
        $bodyParts += $card
    }
}

if ($bodyParts.Count -eq 0) {
    $bodyParts += "<p style='color:#999;padding:20px'>No backup jobs found or accessible with current credentials.</p>"
}

Write-Host "`nBuilding HTML report..." -ForegroundColor Cyan

$html = Build-HTML `
    -Body       ($bodyParts -join "`n") `
    -ServerName $serverName `
    -ReportDate $reportDate `
    -TotalJobs  $jobCount `
    -TotalVMs   $totalVMs

# Resolve output path to same directory as this script
if (-not $OutputPath) {
    $scriptDir  = if ($PSScriptRoot) { $PSScriptRoot } else { Split-Path -Parent $MyInvocation.MyCommand.Path }
    $OutputPath = Join-Path $scriptDir "VeeamBackupReport_$(Get-Date -Format 'yyyyMMdd_HHmmss').html"
}
$OutputPath = [System.IO.Path]::GetFullPath($OutputPath)
$html | Out-File -FilePath $OutputPath -Encoding UTF8 -Force

Write-Host "`n[DONE] Report saved to: $OutputPath" -ForegroundColor Green
Write-Host "       Jobs processed : $jobCount"
Write-Host "       Objects counted: $totalVMs"

# Open in default browser (press Enter to accept default Y, or type N to skip)
$open = Read-Host "`nOpen report in browser? [Y/n]"
if ($open -eq "" -or $open -match '^[Yy]') { Start-Process $OutputPath }

#endregion
